﻿using System;
using System.Diagnostics;
using System.Reflection;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.VBECS.Communication.Channels;
using gov.va.med.VBECS.Communication.Common;
using gov.va.med.VBECS.Communication.Protocols;

namespace gov.va.med.VBECS.Communication.Clients
{
    /// <summary>
    /// Base class for client functionality
    /// </summary>
    /// <typeparam name="T">Pinger object type</typeparam>
#if NUNIT
    public
#else
    internal 
#endif    
    abstract class BaseClient<T> : IClient
        where T : IPinger, new()
    {
        /// <summary>
        /// Default timeout value for connecting a server.
        /// </summary>
        private const int DefaultConnectionAttemptTimeout = 15000; //15 seconds.

        // Logger object
        readonly ILogger _logger = LogManager.Instance().LoggerLocator.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);

        /// <summary>
        /// Pinger object
        /// </summary>
        private readonly T _pinger;

        /// <summary>
        /// The communication channel that is used by client to send and receive messages.
        /// </summary>
        private IChannel _communicationChannel;

        private volatile bool _isDisposed;
        private IProtocol _protocol;

        /// <summary>
        /// Constructor.
        /// </summary>
        // TODO: maybe delete default constructor and remove ProtocolManager usage.
        protected BaseClient()
            : this( ProtocolManager.Instance().ProtocolFactory.CreateProtocol() )
        {}

        /// <summary>
        /// Constructor.
        /// </summary>
        protected BaseClient( IProtocol protocol )
        {
            ConnectTimeout = DefaultConnectionAttemptTimeout;
            Protocol = protocol;
            // create pinger object.
            _pinger = new T();
        }


        #region IClient Members

        /// <summary>
        /// This event is raised when a new message is received.
        /// </summary>
        public event EventHandler<MessageEventArgs> MessageReceived;

        /// <summary>
        /// This event is raised when a new message is sent without any error.
        /// It does not guaranties that message is properly handled and processed by remote application.
        /// </summary>
        public event EventHandler<MessageEventArgs> MessageSent;

        /// <summary>
        /// This event is raised when communication channel closed.
        /// </summary>
        public event EventHandler Connected;

        /// <summary>
        /// This event is raised when client disconnected from server.
        /// </summary>
        public event EventHandler Disconnected;


        /// <summary>
        /// Timeout for connecting to a server (as milliseconds).
        /// Default value: 15 seconds (15000 ms).
        /// </summary>
        public int ConnectTimeout { get; set; }

        /// <summary>
        /// Gets/sets wire protocol that is used while reading and writing messages.
        /// </summary>
        public IProtocol Protocol
        {
            get { return _protocol; }
            set
            {
                if (CommunicationStatus == CommunicationStatus.Connected)
                {
                    throw new ApplicationException("The protocol can not be changed while connected to server.");
                }

                _protocol = value;
            }
        }


        /// <summary>
        /// Gets the communication state of the Client.
        /// </summary>
        public CommunicationStatus CommunicationStatus
        {
            get
            {
                return _communicationChannel != null
                           ? _communicationChannel.Status
                           : CommunicationStatus.Disconnected;
            }
        }

        /// <summary>
        /// Gets the time of the last successfully received message.
        /// </summary>
        public DateTime LastReceivedTime
        {
            get
            {
                return _communicationChannel != null
                           ? _communicationChannel.LastReceivedTime
                           : DateTime.MinValue;
            }
        }

        /// <summary>
        /// Gets the time of the last successfully received message.
        /// </summary>
        public DateTime LastSentTime
        {
            get
            {
                return _communicationChannel != null
                           ? _communicationChannel.LastSentTime
                           : DateTime.MinValue;
            }
        }


        /// <summary>
        /// Connects to server.
        /// </summary>
        public void Connect()
        {
            Protocol.Reset();
            _communicationChannel = CreateCommunicationChannel();
            _communicationChannel.Protocol = Protocol;
            _communicationChannel.Disconnected += communication_channel_disconnected;
            _communicationChannel.MessageReceived += communication_channel_message_received;
            _communicationChannel.MessageSent += communication_channel_message_sent;
            _communicationChannel.Start();
            // start pinger object
            _pinger.Start(this);
            FireConnectedEvent();
        }

        /// <summary>
        /// Disconnects from server.
        /// Does nothing if already disconnected.
        /// </summary>
        public void Disconnect()
        {
            if (CommunicationStatus != CommunicationStatus.Connected)
            {
                return;
            }

            _communicationChannel.Disconnect();
        }

        /// <summary>
        /// Sends a message to the server.
        /// </summary>
        /// <param name="aMessage">Message to be sent</param>
        /// <exception cref="CommunicationException">Throws a CommunicationStateException if client is not connected to the server.</exception>
        public void Send(IMessage aMessage)
        {
            if (CommunicationStatus != CommunicationStatus.Connected)
            {
                throw new CommunicationException("Client is not connected to the server.");
            }

            _communicationChannel.Send(aMessage);
        }

        /// <summary>
        /// Disposes this object and closes underlying connection.
        /// </summary>
        public void Dispose()
        {
            try
            {
                Dispose(true);
            }
#if TRACE
            catch (Exception e)
            {
                Trace.WriteLine(e.ToString());
            }
#else
            catch { }
#endif
            _isDisposed = true;
            GC.SuppressFinalize(this);
        }

        #endregion

        ~BaseClient()
        {
            Dispose(false);
        }


        /// <summary>
        /// This method is implemented by derived classes to create appropriate communication channel.
        /// </summary>
        /// <returns>Ready communication channel to communicate</returns>
        protected abstract IChannel CreateCommunicationChannel();


        /// <summary>
        /// Handles MessageReceived event of _communicationChannel object.
        /// </summary>
        /// <param name="sender">Source of event</param>
        /// <param name="e">Event arguments</param>
        private void communication_channel_message_received(object sender, MessageEventArgs e)
        {
            // If ping message is received and current pinger is MockPinger that means that other pinger functionality applied somewhere else
            // Let ping message to be fired for further processing.
            if (e.Message is IPingMessage && !(_pinger is MockPinger))
            {
                _pinger.PingMessageReceived(e.Message as IPingMessage);
                return;
            }

            FireMessageReceivedEvent(e.Message);
        }

        /// <summary>
        /// Handles MessageSent event of _communicationChannel object.
        /// </summary>
        /// <param name="sender">Source of event</param>
        /// <param name="e">Event arguments</param>
        private void communication_channel_message_sent(object sender, MessageEventArgs e)
        {
            FireMessageSentEvent(e.Message);
        }

        /// <summary>
        /// Handles Disconnected event of _communicationChannel object.
        /// </summary>
        /// <param name="sender">Source of event</param>
        /// <param name="e">Event arguments</param>
        private void communication_channel_disconnected(object sender, EventArgs e)
        {
            _logger.Debug("Communication channel disconnected");
            _pinger.Stop();
            FireDisconnectedEvent();
        }

        /// <summary>
        /// Raises Connected event.
        /// </summary>
        protected virtual void FireConnectedEvent()
        {
            if (Connected != null)
            {
                Connected.Invoke(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Raises Disconnected event.
        /// </summary>
        protected virtual void FireDisconnectedEvent()
        {
            if (Disconnected != null)
            {
                Disconnected.Invoke(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Raises MessageReceived event.
        /// </summary>
        /// <param name="theMessage">Received message</param>
        protected virtual void FireMessageReceivedEvent(IMessage theMessage)
        {
            if (MessageReceived != null)
            {
                MessageReceived.Invoke(this, new MessageEventArgs(theMessage));
            }
        }

        /// <summary>
        /// Raises MessageSent event.
        /// </summary>
        /// <param name="theMessage">Received message</param>
        protected virtual void FireMessageSentEvent(IMessage theMessage)
        {
            if (MessageSent != null)
            {
                MessageSent.Invoke(this, new MessageEventArgs(theMessage));
            }
        }

        protected virtual void Dispose(bool disposing)
        {
            Disconnect();
        }
    }
}